<?php
namespace app\common\controller;
use app\api\model\User AS UserModel;
use think\Loader;
use think\Request;
use think\Response;
use think\exception\HttpResponseException;
/**
 * 前台API总控制器
 */
class ApiBase{
	protected $request;
	protected $webdb;//网站配置信息
	protected $user;//用户登录后的信息
	/**
	 * 默认响应输出类型,支持json/xml
	 * @var string
	 */
	protected $responseType='json';
	/**
	 * 无需登录的方法
	 * @var array
	 */
	protected $noNeedLogin=[];
	/**
	 * 过期时间秒数 默认5分钟
	 * @var int
	 */
	protected $expires=300;
	/**
	 * 登录过期时间 默认30天
	 * @var float|int
	 */
	protected $login_time=2592000;
	/**
	 * 构造方法
	 * @access public
	 * @param Request $request Request 对象
	 */
	public function __construct(Request $request=null){
		$this->request=is_null($request)?Request::instance():$request;
		// 控制器初始化
		$this->_initialize();
	}
	/**
	 * 初始化
	 */
	protected function _initialize(){
		define('LOAD_INDEXBASE',false);//跳过首页的钩子
		$this->check_Token($this->request->only(['accesstoken','timestamp']));
		//路由信息
		$modulename=$this->request->module();
		$controllername=Loader::parseName($this->request->controller());
		$actionname=strtolower($this->request->action());
		$this->webdb=config('webdb'); //网站配置信息
		//用户登录信息，如果没登录的话，就为空值
		$this->user=UserModel::login_info();
		// 检测是否需要验证登录
		if(!$this->match($this->noNeedLogin)){
			if(!$this->user){
				$this->error('登录后操作');
			}
		}
	}
	/**
	 * 简单的接口权限判断
	 */
	public function check_Token($array=[]){
		if(strlen(config("webdb.M__api")['api_secret'])<32){
			$this->error('请配置API密匙或者密匙小于32位');
		}
		if(!isset($array['timestamp'])||intval($array['timestamp'])<=1){
			$this->error('时间戳不存在');
		}
		if(time()-intval($array['timestamp'])>$this->expires){
			$this->error('请求超时');
		}
		//加密规则 传递的时间戳+密匙
		$service_token=md5($array['timestamp'].config("webdb.M__api")['api_secret']);
		if($array['accesstoken']!==$service_token){
			$this->error('accesstoken不正确');
		}
	}
	/**
	 * 检测当前控制器和方法是否匹配传递的数组
	 * @param array $arr 需要验证权限的数组
	 * @return boolean
	 */
	public function match($arr=[]){
		$request=Request::instance();
		$arr=is_array($arr)?$arr:explode(',',$arr);
		if(!$arr){
			return false;
		}
		$arr=array_map('strtolower',$arr);
		// 是否存在
		if(in_array(strtolower($request->action()),$arr)||in_array('*',$arr)){
			return true;
		}
		// 没找到匹配
		return false;
	}
	/**
	 * 操作成功返回的数据
	 * @param string $msg 提示信息
	 * @param mixed $data 要返回的数据
	 * @param int $code 错误码，默认为200
	 * @param string $type 输出类型
	 * @param array $header 发送的 Header 信息
	 */
	protected function success($msg='',$data=null,$code=200,$type=null,array $header=[]){
		$this->result($msg,$data,$code,$type,$header);
	}
	/**
	 * 操作失败返回的数据
	 * @param string $msg 提示信息
	 * @param mixed $data 要返回的数据
	 * @param int $code 错误码，默认为400
	 * @param string $type 输出类型
	 * @param array $header 发送的 Header 信息
	 */
	protected function error($msg='',$data=null,$code=400,$type=null,array $header=[]){
		$this->result($msg,$data,$code,$type,$header);
	}
	/**
	 * 返回封装后的 API 数据到客户端
	 * @access protected
	 * @param mixed $msg 提示信息
	 * @param mixed $data 要返回的数据
	 * @param int $code 错误码，默认为0
	 * @param string $type 输出类型，支持json/xml/jsonp
	 * @param array $header 发送的 Header 信息
	 * @return void
	 * @throws HttpResponseException
	 */
	protected function result($msg,$data=null,$code=0,$type=null,array $header=[]){
		$result=[
			'code'=>$code,
			'msg' =>$msg,
			'time'=>Request::instance()->server('REQUEST_TIME'),
			'data'=>$data,
		];
		// 如果未设置类型则自动判断
		$type=$type?$type:($this->request->param(config('var_jsonp_handler'))?'jsonp':$this->responseType);
		if(isset($header['statuscode'])){
			$code=$header['statuscode'];
			unset($header['statuscode']);
		}else{
			//未设置状态码,根据code值判断
			$code=$code>=1000||$code<200?200:$code;
		}
		$response=Response::create($result,$type,$code)->header($header);
		throw new HttpResponseException($response);
	}
	protected function ok_js($data=[],$msg='操作成功',$page_rows=0){
		header("Access-Control-Allow-Origin:*");
		header("Access-Control-Allow-Methods:GET,POST");
		if(is_string($data)||is_numeric($data)){
			if(input('debug')){ //调试,查看原始数据
				return $data;
			}
			return json([
				'code'=>0,
				'msg' =>$msg,
				'data'=>$data,
			]);
		}
		$array=$data=getArray($data);
		if(isset($array['data'])&&isset($array['total'])){
			$data=$array['data'];
		}elseif(isset($array['data'])&&empty($array['data'])){
			$data='';
		}else{
			$array=['current_page'=>1,'last_page'=>1,'per_page'=>$page_rows,'current_page'=>1,'current_page'=>1];
			$array['total']=$array['per_page']=count($data);
		}
		$hasNext=false;
		$next=$array['current_page'];
		if($array['current_page']<$array['last_page']){
			$hasNext=true;
			$next++;
		}
		$_array=[];
		if(is_array($array)){ //主要是服务于标签.因为标签中data全是html网页代码字符串,导致不能传递更多的数据
			$_array=$array;
			unset($_array['data'],$_array['page'],$_array['pages'],$_array['perPage'],$_array['total'],$_array['prev'],$_array['next'],$_array['hasNext'],$_array['hasPrev'],$_array['per_page'],$_array['current_page'],$_array['last_page']);
		}
		$array=[
			'code'    =>0,
			'msg'     =>$msg,
			'data'    =>$data,
			'ext'     =>$_array,
			'paginate'=>[
				'page'   =>empty($array['data'])?input('page'):$array['current_page'],
				//当前页码, 要特别注意,系统分页函数,当数据不存在的时候,不会显示真实页码
				'pages'  =>$array['last_page'],
				//总页数
				'perPage'=>$array['per_page'],
				//每页几条
				'total'  =>$array['total'],
				//总共几条
				'prev'   =>$array['current_page'],
				//上一页的页码
				'next'   =>$next,
				//下一页的页码
				'hasNext'=>$hasNext,
				//下一页是否存在
				'hasPrev'=>$array['current_page']>1?true:false,
				//上一页是否存在
			],
		];
		if(input('debug')){ //调试,查看原始数据
			print_r($array);
			return;
		}
		return json($array);
	}
	protected function err_js($msg='操作失败',$data=[],$code=1){
		header("Access-Control-Allow-Origin:*");
		header("Access-Control-Allow-Methods:GET,POST");
		$array=[
			'code'=>$code,
			'msg' =>$msg,
			'data'=>$data,
		];
		if(input('debug')){ //调试,查看原始数据
			print_r($array);
			return;
		}
		return json($array);
	}
	/**
	 * 齐博首创 钩子文件扩展接口
	 * 详细使用教程 https://www.kancloud.cn/php168/x1_of_qibo/1010065
	 * @param string $type 钩子标志,不能重复
	 * @param array $data POST表单数据 可以改变其值
	 * @param array $info 数据库资料
	 * @param array $array 其它参数
	 * @param string $use_common 默认同时调用全站通用的
	 * @return unknown|NULL
	 */
	protected function get_hook($type='',&$data=[],$info=[],$array=[],$use_common=true){
		$path_array=[];
		preg_match_all('/([_a-z]+)/',get_called_class(),$carray);
		$dirname=$carray[0][1];
		$path_array[]=(defined('IN_PLUGIN')?PLUGINS_PATH:APP_PATH).$dirname.DS.'ext'.DS.$type.DS;
		if($use_common===true){
			$path_array[]=APP_PATH.'common'.DS.'ext'.DS.$type.DS;
		}
		$file_array=[];
		foreach($path_array AS $path){
			if(is_dir($path)){
				$sarray=[];
				$dir=opendir($path);
				while($file=readdir($dir)){
					if(preg_match("/^([\w\.-]*)\.php$/i",$file,$sar)){
						if(in_array($sar[1],$file_array)){
							continue; //出现同名,就跳过
						}
						$sarray[$path.DS.$file]=$sar[1];
					}
				}
				asort($sarray);
				$file_array=array_merge($file_array,$sarray);
			}
		}
		if($file_array){
			foreach($file_array AS $file=>$v){
				$result=include($file);
				if($result===true||$result===false){
					return $result;
				}elseif(is_string($result)||is_array($result)){
					return $result;
				}
			}
		}
		return null;
	}
}
